<?php
/** Ultra-minimal CSV importer for the Renewals schema. 
 * Usage:
 * 1) Upload renewals_clean.csv next to this file.
 * 2) Open in browser: quick_import.php (or quick_import.php?file=your.csv)
 */
ini_set('display_errors', '1');
error_reporting(E_ALL);
require __DIR__ . '/db.php';

$csv = $_GET['file'] ?? 'renewals_clean.csv';
$path = __DIR__ . '/' . $csv;
if (!file_exists($path)) {
    http_response_code(404);
    echo "CSV not found at " . htmlspecialchars($path);
    exit;
}

$pdo = db();
$pdo->beginTransaction();

$fp = fopen($path, 'r');
$header = fgetcsv($fp);
$cols = array_map('trim', $header);

// locate columns
function find_col($cols, $cands) {
    foreach ($cands as $c) {
        $i = array_search($c, $cols);
        if ($i !== false) return $i;
    }
    return null;
}

$idx_domain  = find_col($cols, ['DOMAIN','Domian','WEBSITE','URL','HOSTNAME']);
$idx_account = find_col($cols, ['ACCOUNT','CLIENT','CUSTOMER','COMPANY','ACCOUNT_NAME']);
$idx_service = find_col($cols, ['SERVICE','SERVICES','PRODUCT','PRODUCTS']);
$idx_price   = find_col($cols, ['PRICE','TOTAL','AMOUNT','COST']);
$idx_month   = find_col($cols, ['MONTH','RENEWAL_MONTH','DUE_MONTH']);

// year columns
$year_idx = [];
for ($i=0; $i<count($cols); $i++) {
    if (preg_match('/RENEWAL[_ ]?YEAR\s*(\d{4})/i', $cols[$i], $m)) {
        $year_idx[$i] = (int)$m[1];
    } elseif (preg_match('/^(20\d{2})$/', $cols[$i], $m)) {
        $year_idx[$i] = (int)$m[1];
    }
}

$created_accounts = $created_servers = $attached_services = $renewals = 0;

function month_to_int($m) {
    if ($m === null) return null;
    $m = trim($m);
    if ($m === '') return null;
    if (ctype_digit($m)) {
        $n = (int)$m;
        return ($n>=1 && $n<=12) ? $n : null;
    }
    $map = [
        'JANUARY'=>1,'FEBRUARY'=>2,'MARCH'=>3,'APRIL'=>4,'MAY'=>5,'JUNE'=>6,
        'JULY'=>7,'AUGUST'=>8,'SEPTEMBER'=>9,'OCTOBER'=>10,'NOVEMBER'=>11,'DECEMBER'=>12
    ];
    $key = strtoupper(preg_replace('/\s+/', '', $m));
    foreach ($map as $name=>$num) {
        if (strpos($name, $key) === 0 || strpos($key, $name) === 0) return $num;
    }
    return null;
}

while (($row = fgetcsv($fp)) !== false) {
    $domain  = $idx_domain  !== null ? trim($row[$idx_domain] ?? '') : '';
    $account = $idx_account !== null ? trim($row[$idx_account] ?? '') : '';
    $service = $idx_service !== null ? trim($row[$idx_service] ?? '') : '';
    $price   = $idx_price   !== null ? trim($row[$idx_price] ?? '') : '';
    $monthV  = $idx_month   !== null ? trim($row[$idx_month] ?? '') : '';

    if ($domain === '' and $account === '') continue; // skip empty row
    if ($account === '') $account = 'Unknown';

    // 1) ensure account
    $aid = null;
    $q = $pdo->prepare("SELECT id FROM accounts WHERE name=?");
    $q->execute([$account]);
    $aid = $q->fetchColumn();
    if (!$aid) {
        $pdo->prepare("INSERT INTO accounts (name, created_at, updated_at) VALUES (?, NOW(), NOW())")->execute([$account]);
        $aid = (int)$pdo->lastInsertId();
        $created_accounts++;
    }

    // 2) ensure server (domain)
    if ($domain === '') $domain = 'no-domain-' . uniqid();
    $sid = null;
    $q = $pdo->prepare("SELECT id FROM servers WHERE account_id=? AND domain=?");
    $q->execute([$aid, $domain]);
    $sid = $q->fetchColumn();
    $month_i = month_to_int($monthV) ?? 1;

    if (!$sid) {
        $override = null;
        if ($price !== '' && is_numeric($price)) $override = (float)$price;
        $pdo->prepare("INSERT INTO servers (account_id, domain, month, price_override, auto_price, created_at, updated_at) VALUES (?,?,?,?,0, NOW(), NOW())")
            ->execute([$aid, $domain, $month_i, $override]);
        $sid = (int)$pdo->lastInsertId();
        $created_servers++;
    } else {
        // Optional: update month/override on existing
        if ($price !== '' && is_numeric($price)) {
            $pdo->prepare("UPDATE servers SET month=?, price_override=?, updated_at=NOW() WHERE id=?")
                ->execute([$month_i, (float)$price, $sid]);
        } else {
            $pdo->prepare("UPDATE servers SET month=?, updated_at=NOW() WHERE id=?")
                ->execute([$month_i, $sid]);
        }
    }

    // 3) attach services (split by + or ,)
    if ($service !== '') {
        $parts = preg_split('/[+,]/', $service);
        foreach ($parts as $p) {
            $name = trim($p);
            if ($name === '') continue;
            // ensure product
            $pid = null;
            $q = $pdo->prepare("SELECT id, base_price FROM products WHERE name=?");
            $q->execute([$name]);
            $pr = $q->fetch();
            if (!$pr) {
                $pdo->prepare("INSERT INTO products (name, base_price, created_at, updated_at) VALUES (?, 0.00, NOW(), NOW())")->execute([$name]);
                $pid = (int)$pdo->lastInsertId();
                $unit = 0.00;
            } else {
                $pid = (int)$pr['id'];
                $unit = (float)$pr['base_price'];
            }
            // attach
            $pdo->prepare("INSERT INTO server_services (server_id, product_id, quantity, unit_price, line_total, created_at, updated_at) VALUES (?,?,?,?,?, NOW(), NOW())")
                ->execute([$sid, $pid, 1, $unit, $unit]);
            $attached_services++;
        }
        // recalc auto_price based on server_services
        $pdo->prepare("UPDATE servers s SET auto_price=(SELECT COALESCE(SUM(line_total),0) FROM server_services WHERE server_id=s.id) WHERE s.id=?")->execute([$sid]);
    }

    // 4) renewals (per year columns)
    foreach ($year_idx as $pos=>$year) {
        $val = trim((string)($row[$pos] ?? ''));
        $status = (strtoupper($val) === 'PAID') ? 'PAID' : 'UNPAID';
        // upsert
        $q = $pdo->prepare("SELECT id FROM renewals WHERE server_id=? AND year=?");
        $q->execute([$sid, $year]);
        $rid = $q->fetchColumn();
        $amount = 0.0;
        $q2 = $pdo->prepare("SELECT COALESCE(price_override, auto_price) AS tot, month FROM servers WHERE id=?");
        $q2->execute([$sid]);
        $sv = $q2->fetch();
        $amount = (float)$sv['tot'];
        $due_m  = (int)$sv['month'];
        if (!$rid) {
            $pdo->prepare("INSERT INTO renewals (server_id, year, status, amount_due, due_month, created_at, updated_at) VALUES (?,?,?,?,?, NOW(), NOW())")
                ->execute([$sid, $year, $status, $amount, $due_m]);
        } else {
            $pdo->prepare("UPDATE renewals SET status=?, amount_due=?, due_month=?, updated_at=NOW() WHERE id=?")
                ->execute([$status, $amount, $due_m, $rid]);
        }
        $renewals++;
    }
}

fclose($fp);
$pdo->commit();

echo "<pre>Import complete.\n";
echo "Accounts created: $created_accounts\n";
echo "Servers created: $created_servers\n";
echo "Services attached: $attached_services\n";
echo "Renewal rows upserted: $renewals\n";
echo "</pre>";
